/*  -*- pse-c -*-
 *----------------------------------------------------------------------------
 * Filename: iegd_interface.c
 * $Revision: 1.1.2.8 $
 *----------------------------------------------------------------------------
 * Gart and DRM driver for Intel Embedded Graphics Driver
 * Copyright © 2007, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 
 * 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include "global.h"
#include "intelpci.h"

static int iegd_nap_fetch_size(void);
static void iegd_nap_tlbflush(struct agp_memory *mem);

static void iegd_iq35_init_gtt_entries(void);
static void iegd_nap_iq35_gatt(void);
static int iegd_nap_9series(u32 order);
static int AGP_CREATE_GATT(iegd_nap_create_gatt_table);
static void iegd_nap_cleanup(void);


struct aper_size_info_fixed iegd_i915_sizes[] =
{
	{128, 32768, 5},
	/* The 64M mode still requires a 128k gatt */
	{64, 16384, 5},
	{256, 65536, 6},
	{512, 131072, 7},
};

struct aper_size_info_fixed iegd_iq35_sizes[] =
{
	{128, 32768, 5},
	{256, 65536, 6},
	{512, 131072, 7},
};

bridge_driver_t drv_nap = {
	.owner              = THIS_MODULE,
	.size_type          = FIXED_APER_SIZE,
	.aperture_sizes     = 0,
	.num_aperture_sizes = 0,
	.needs_scratch_page = TRUE,
	.configure          = iegd_cmn_configure,
	.fetch_size         = iegd_nap_fetch_size,
	.cleanup            = iegd_nap_cleanup,
	.tlb_flush          = iegd_nap_tlbflush,
	.mask_memory        = iegd_cmn_mask_memory,
	.masks              = iegd_cmn_masks,
	.agp_enable         = iegd_cmn_agp_enable,
	.cache_flush        = global_cache_flush,
	.create_gatt_table  = iegd_nap_create_gatt_table,
	.free_gatt_table    = iegd_cmn_free_gatt_table,
	.insert_memory      = iegd_cmn_insert_entries,
	.remove_memory      = iegd_cmn_remove_entries,
	.alloc_by_type      = iegd_cmn_alloc_by_type,
	.free_by_type       = iegd_cmn_free_by_type,
	.agp_alloc_page     = agp_generic_alloc_page,
	.agp_destroy_page   = agp_generic_destroy_page,
};

static int iegd_nap_fetch_size()
{
	struct aper_size_info_fixed *values;
	u32 offset = 0;
	u32 temp2;
	u8 temp;

#define IQ35_GMCH_MSAC 0x62
#define I915_256MB_ADDRESS_MASK (1<<27)

	AGN_DEBUG("Enter");

	values = A_SIZE_FIX(agp_bridge->driver->aperture_sizes);

	switch(private_data.pdev->device) {
	case PCI_DEVICE_ID_Q35:
		pci_read_config_byte(private_data.pdev,
			IQ35_GMCH_MSAC, &temp);
		switch(temp & 0x3) {
		case 1:
			offset = 2; /* 512MB aperture size */
			break;
		case 2:
			offset = 1; /* 256MB aperture size */
			break;
		case 3:
			offset = 0; /* 128MB aperture size */
			break;
		}
		break;
	case PCI_DEVICE_ID_915GD:
	case PCI_DEVICE_ID_915AL:
	case PCI_DEVICE_ID_945G:
	case PCI_DEVICE_ID_945GM:
	case PCI_DEVICE_ID_945GME:
		pci_read_config_dword(private_data.pdev,
			I915_GMADDR, &temp2);
		if (temp2 & I915_256MB_ADDRESS_MASK) {
			offset = 0;	/* 128MB aperture */
		} else {
			offset = 2;	/* 256MB aperture */
		}
		break;
	}

	agp_bridge->previous_size = agp_bridge->current_size =
		(void *)(values + offset);

	AGN_DEBUG("Exit");

	return values[offset].size;
}

static void iegd_nap_tlbflush(struct agp_memory *mem)
{
	AGN_DEBUG("Enter");
	return;
	AGN_DEBUG("Exit");
}

static void iegd_iq35_init_gtt_entries(void)
{
	u16 gmch_ctrl;
	u32 iegd_scratch, iegd_scratch2;
	int gtt_entries;
	int local = 0;
	int size = 4;

#define I35_GMCH_GMS_STOLEN_128M   (0x8 << 4)
#define I35_GMCH_GMS_STOLEN_256M   (0x9 << 4)
#define I35_GMCH_GMS_MASK          0xf0

	AGN_DEBUG("Enter");

	pci_read_config_word(agp_bridge->dev,I830_GMCH_CTRL,&gmch_ctrl);

	switch (gmch_ctrl & I35_GMCH_GMS_MASK) {
		case I855_GMCH_GMS_STOLEN_1M:
			gtt_entries = MB(1) - KB(size);
			break;
		case I855_GMCH_GMS_STOLEN_4M:
			gtt_entries = MB(4) - KB(size);
			break;
		case I855_GMCH_GMS_STOLEN_8M:
			gtt_entries = MB(8) - KB(size);
			break;
		case I855_GMCH_GMS_STOLEN_16M:
			gtt_entries = MB(16) - KB(size);
			break;
		case I855_GMCH_GMS_STOLEN_32M:
			gtt_entries = MB(32) - KB(size);
			break;
		case I915_GMCH_GMS_STOLEN_48M:
			gtt_entries = MB(48) - KB(size);
			break;
		case I915_GMCH_GMS_STOLEN_64M:
			gtt_entries = MB(64) - KB(size);
			break;
		case I35_GMCH_GMS_STOLEN_128M:
			gtt_entries = MB(128) - KB(size);
			break;
		case I35_GMCH_GMS_STOLEN_256M:
			gtt_entries = MB(256) - KB(size);
			break;
		default:
			gtt_entries = 0;
		break;
	}

	iegd_scratch = readl(private_data.registers + 0x71410);

	/* FIXME: check for the pci card as primary */
	if(iegd_scratch == 0) {
		gtt_entries = 0;
	} else if (((iegd_scratch>>16) == 0xE1DF) && (iegd_scratch & 0x4)) {
		AGN_LOG("IEGD Firmware Detected");
		/* IEGD firmware found, and Mem Reservation Flag present */
		iegd_scratch2 = readl(private_data.registers + 0x71418);
		gtt_entries = (iegd_scratch2 & 0xFFFF) * 4096;
	}

	if (gtt_entries > 0) {
		AGN_LOG("Detected %dK %s memory.",
		gtt_entries / KB(1), local ? "local" : "stolen");
	} else {
		AGN_LOG("No pre-allocated video memory detected.");
	}

	gtt_entries /= KB(4);

	private_data.gtt_entries = gtt_entries;

	AGN_DEBUG("Exit");
}

static void iegd_nap_iq35_gatt()
{
	u32 gtt_mem_size;
	u32 base_stolen_mem;
	u16 gmch_ctrl;

	AGN_DEBUG("Enter");

	iegd_iq35_init_gtt_entries();

	pci_read_config_dword(private_data.pdev,
		IQ35_BASE_STOLEN, &base_stolen_mem);
	base_stolen_mem &= 0xFFF00000;

	pci_read_config_word(private_data.pdev,
		I830_GMCH_CTRL, &gmch_ctrl);

	switch(gmch_ctrl & IQ35_GTT_MEM_SIZE) {
	case IQ35_GGMS_1MB:
		gtt_mem_size = MB(1); /* Normal mode */
		break;
	case IQ35_GGMS_2MB:
		gtt_mem_size = MB(2); /* VT mode */
		break;
	default:
		gtt_mem_size = 0;
	}

	AGN_DEBUG("gtt_mem_size = %uMB", gtt_mem_size);

	/* Minus based stolen memory to get the base of gtt. This address
	 * can also get from register 0xA8 of config space device 0 */
	agp_bridge->gatt_bus_addr = base_stolen_mem - gtt_mem_size;

	AGN_DEBUG("Exit");
}

static int iegd_nap_9series(u32 order)
{
	u32 gtt_pgctl_reg;
	u32 gtt_bus_addr;
	u32 gtt_enabled = FALSE;
	int num_entries;
	char *gtt_table, *gtt_table_end, *current_entry;
	struct page *gtt_table_page;
	int i;
	u16 j = 0;

	AGN_DEBUG("Enter");

	gtt_pgctl_reg = readl(private_data.registers +
		I810_PGETBL_CTL);
	global_cache_flush();
	gtt_bus_addr = gtt_pgctl_reg & 0xFFFFF000;
	gtt_enabled  = gtt_pgctl_reg & I810_PGETBL_ENABLED;

	/* we have to call this as early as possible after the MMIO base
	 * address is known */
	iegd_cmn_init_gtt_entries();

	/*
	 * If GTT not enabled created our own gtt table from kernel memory
	 * and initialize it to scratch page. This in case the VBIOS is
	 * not our VBIOS
	 */
	if (!gtt_enabled) {
		num_entries = intel_i830_sizes[0].num_entries;
		gtt_table   = (char *)__get_free_pages(GFP_KERNEL,
			order);
		gtt_table_end = gtt_table +
			(1 << order) * PAGE_SIZE - 1;

		/* Make sure allocation was successful */
		if (NULL == gtt_table) {
			AGN_ERROR("Fail to allocate kernel pages");
			return (-ENOMEM);
		}

		for (current_entry = gtt_table; current_entry < gtt_table_end;
			current_entry += PAGE_SIZE ) {
			gtt_table_page = virt_to_page( current_entry );
			set_bit(PG_reserved, &gtt_table_page->flags);
		}

		agp_bridge->gatt_bus_addr = virt_to_phys(gtt_table);

		for (i = 0; i < num_entries; i++) {
			gtt_table[j] = (unsigned long) agp_bridge->scratch_page;
			j += 4;
		}
	} else {
		agp_bridge->gatt_bus_addr = gtt_bus_addr;
	}

	AGN_DEBUG("Exit");

	return 0;
}


static int AGP_CREATE_GATT(iegd_nap_create_gatt_table)
{
	const u32 i915_gtt_table_order = 6;
	u32 mmio_bus_addr, temp2;
	int ret;

	AGN_DEBUG("Enter");

	agp_bridge->gatt_table_real = NULL;

	/* Find and save the address of the MMIO register */
	pci_read_config_dword(private_data.pdev, I915_MMADDR,
		&mmio_bus_addr);
	mmio_bus_addr &= 0xFFF80000;

	private_data.registers = (volatile u8 *) ioremap(mmio_bus_addr,
		128 * 4096);
	if (!private_data.registers) {
		AGN_ERROR("ioremap failed to map mmio");
		return (-ENOMEM);
	}

	pci_read_config_dword(private_data.pdev, I915_PTEADDR,&temp2);

	/* FIXME: double check the size of area to map to pci space */
	private_data.gtt = (volatile u32 *)ioremap(temp2, 256 * 1024);
	if (!private_data.gtt) {
		AGN_ERROR("ioremap failed to map gtt");
		return (-ENOMEM);
	}

	switch(private_data.pdev->device) {
	case PCI_DEVICE_ID_Q35:
		/* Bearlake B is difference from other chipset, especially
		 * when reading gtt based address. Probably future chipset
		 * will have same architecture as Bearlake-B and this
		 * code can move to common file*/
		iegd_nap_iq35_gatt();
		break;
	case PCI_DEVICE_ID_915GD:
	case PCI_DEVICE_ID_915AL:
	case PCI_DEVICE_ID_945G:
	case PCI_DEVICE_ID_945GM:
	case PCI_DEVICE_ID_945GME:
		if((ret = iegd_nap_9series(i915_gtt_table_order))) {
			return (ret);
		}
		break;
	}

	agp_bridge->gatt_table = NULL;

	AGN_DEBUG("Exit");

	return (0);
}

static void iegd_nap_cleanup(void)
{

	AGN_DEBUG("Enter");
	iounmap((void *)private_data.gtt);
	iounmap((void *)private_data.registers);
	AGN_DEBUG("Exit");
}

